/*
 * GTBE_adcLibTest.c - Tests out the functionality of the TI ads1278 driver library
 *  Utilizes the Georgia Tech Backend (GTBE) Hardware
 *
 *  Author: Curtis Mayberry
 *  Georgia Tech IMEMS
 *  March 2014
 *
 *  GT BE Peripherals:
 *   ADC: ADS1278
 *    INPUTS
 */

#include <stdint.h>
#include <stdbool.h>
#include <math.h>
#include "inc/hw_memmap.h"
#include "inc/hw_ssi.h"
#include "inc/hw_types.h"
#include "inc/hw_ints.h"

// Tivaware
#include "driverlib/rom.h"
#include "driverlib/ssi.h"
#include "driverlib/sysctl.h"
#include "driverlib/gpio.h"
#include "driverlib/pin_map.h"
#include "driverlib/fpu.h"
#include "driverlib/pwm.h"
#include "driverlib/interrupt.h"
#include "driverlib/uart.h"
//#include "driverlib/timer.h"
//#include "driverlib/udma.h"

// Launchpad Drivers
//#include "examples/boards/ek-tm4c123gxl/drivers/rgb.c"

// GTBE Lib
#include "dac_ad5754.h"
#include "adc_ads1278.h"

#ifndef M_PI
#define M_PI                    3.14159265358979323846
#endif

/**************
 * Prototypes *
 **************/
void Timer2IntHandlerDACout(void);

/*********************
 * Hardware Settings *
 *********************/
#define BIN TWOS_COMPLEMENT // DAC data format in bipolar modes

/********************
 * Global Variables *
 ********************/
//#define BUFFER_SIZE 1024
//uint32_t ADCdatabufferByte2[BUFFER_SIZE];
//uint32_t ADCdatabufferByte1[BUFFER_SIZE];
//uint32_t ADCdatabufferByte0[BUFFER_SIZE];
//uint32_t ADC_Reading[BUFFER_SIZE];
//float ADC_ReadingV[BUFFER_SIZE];
//uint32_t bufferIdx = 0;
//uint32_t bufferIdxPrevious = 0;

#define BUFFER_SIZE 1024
// ADC Buffers
uint32_t gADC_dataBufferByte2[BUFFER_SIZE];
uint32_t gADC_dataBufferByte1[BUFFER_SIZE];
uint32_t gADC_dataBufferByte0[BUFFER_SIZE];
uint32_t gADC_reading[BUFFER_SIZE];
uint32_t g_bufferIdx = 0;
uint32_t g_bufferIdxPrevious = 0;
// DAC output buffer
int32_t gDAC_outputBuffer[BUFFER_SIZE];
// Signal Gain
uint32_t A = 2;
// UART Command Handling
int32_t g_UARTCommand;

/******************
 * GPIO Functions *
 ******************/

/**
 * An Interrupt handler which executes each time ~DRDY goes low.  The
 **/
void intHandlerDRDY(void) {
uint32_t intStatus;
	intStatus = GPIOIntStatus(GPIO_PORTB_BASE, true);
	GPIOIntClear(GPIO_PORTB_BASE, intStatus);
	if(intStatus == GPIO_INT_PIN_5) {
		// Receive Each byte over SSI
		ROM_SSIDataPut(SSI2_BASE,0x00000000); // Byte 2
		ROM_SSIDataPut(SSI2_BASE,0x00000000); // Byte 1
		ROM_SSIDataPut(SSI2_BASE,0x00000000); // Byte 0

		// Transfer each byte to the buffers
		ROM_SSIDataGet(SSI2_BASE,&gADC_dataBufferByte2[g_bufferIdx]);
		ROM_SSIDataGet(SSI2_BASE,&gADC_dataBufferByte1[g_bufferIdx]);
		ROM_SSIDataGet(SSI2_BASE,&gADC_dataBufferByte0[g_bufferIdx]);

		if(g_bufferIdx > 0) {
			DACd_updateDataDig(DAC_ADDR_A | DAC_ADDR_NONE_EH, gDAC_outputBuffer[g_bufferIdx], 0x00000000);
		}

		g_bufferIdx++;

		if(g_bufferIdx == BUFFER_SIZE) {
			g_bufferIdx = 0;
		}
	}
}

/******************
 * FPU Functions *
 ******************/
/**
 * Initializes the FPU
 **/
void setupFPU(void) {
	// Lazy Stacking increases interrupt latency and stack usage
	//  (only need if doing floating pt. in interrupts)
	//FPULazyStackingEnable();
	ROM_FPUEnable();
}


/******************
 * UART Functions *
 ******************/
/**
 * Initializes the UART
 **/
 void initUART0(void) {
	 ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);
	 ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

	 ROM_GPIOPinConfigure(GPIO_PA0_U0RX);
	 ROM_GPIOPinConfigure(GPIO_PA1_U0TX);
	 ROM_GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

	 ROM_UARTConfigSetExpClk(UART0_BASE, SysCtlClockGet(), 115200,
	 (UART_CONFIG_WLEN_8 | UART_CONFIG_STOP_ONE | UART_CONFIG_PAR_NONE));
 }

/**
 * Initializes the UART interrupt
 */
 void initUartInterrupt(void) {
	 ROM_UARTIntEnable(UART0_BASE, UART_INT_RX);
 }

/**
 * Receives commands for the device
 **/
 void uartMonitorISR(void) {
	 g_UARTCommand = UARTCharGetNonBlocking(UART0_BASE);
 }

/******************
 * NVIC Functions *
 ******************/

/**
 * Sets the priority of the interrupts
 *  The ~DRDY sample ISR is given highest priority
 *  The UART RX ISR is given lower priority
 *  Smaller numbers correspond to higher interrupt priorities; priority 0 is the highest
 *  interrupt priority.
 **/
 initIntPriority(void) {
	 // Highest Priority
	 ROM_IntPrioritySet(INT_GPIOB, 0x00);
	 ROM_IntPrioritySet(INT_SSI0, 0x20);
	 ROM_IntPrioritySet(UART_INT_RX, 0xE0);
	 // Lowest Priority

 }

int main(void) {
	//
	// System Initialization
	//
	// Set system clock to 80 MHz (400MHz main PLL (divided by 5 - uses DIV400 bit)  [16MHz external xtal drives PLL]
	ROM_SysCtlClockSet(SYSCTL_SYSDIV_2_5 | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN | SYSCTL_XTAL_16MHZ);
	//setupFPU();
	initUART0();

	//
	// DAC initialization
	//
	DAC_initDAC(DAC_RANGE_PM5V, DAC_PWR_PUA | DAC_PWR_PUB);
	DAC_initSSI0Int();

	//
	// ADC initialization
	//
	ADC_initADC();
	ADC_initDRDYint();

	// Process data between samples
	while(1) {
		if((g_bufferIdx == 0) && (g_bufferIdxPrevious!= 0)) {
			gADC_reading[BUFFER_SIZE-1] = ((((gADC_dataBufferByte2[BUFFER_SIZE-1]<<16)+(gADC_dataBufferByte1[BUFFER_SIZE-1]<<8)+(gADC_dataBufferByte0[BUFFER_SIZE-1]))) >> 8);
			gDAC_outputBuffer[BUFFER_SIZE-1] = A * gDAC_outputBuffer[BUFFER_SIZE-1];
			g_bufferIdxPrevious = g_bufferIdx;

		}
		else if(g_bufferIdx > g_bufferIdxPrevious) {
			gADC_reading[g_bufferIdx-1] = ((((gADC_dataBufferByte2[g_bufferIdx-1]<<16)+(gADC_dataBufferByte1[g_bufferIdx-1]<<8)+(gADC_dataBufferByte0[g_bufferIdx-1]))) >> 8);
			gDAC_outputBuffer[g_bufferIdx-1] = A * gADC_reading[g_bufferIdx-1];
			g_bufferIdxPrevious = g_bufferIdx;
		}
	}
}
